<?php
/**
 * Klaviyo API Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF7_Klaviyo_API {

    private $api_key;
    private $api_revision;
    private $base_url = 'https://a.klaviyo.com/api/';

    public function __construct() {
        $this->api_key = CF7_Klaviyo_Settings::get_api_key();
        $this->api_revision = CF7_Klaviyo_Settings::get_api_revision();
    }

    /**
     * Make API request to Klaviyo
     */
    private function make_request($endpoint, $method = 'GET', $data = null) {
        $url = $this->base_url . $endpoint;

        $args = array(
            'method' => $method,
            'headers' => array(
                'Authorization' => 'Klaviyo-API-Key ' . $this->api_key,
                'revision' => $this->api_revision,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'timeout' => 30,
            'sslverify' => true,
        );

        if ($data && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = json_encode($data);
        }

        // Log de debug (solo si WP_DEBUG está activo)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[CF7 Klaviyo API] Request: ' . $method . ' ' . $url);
            error_log('[CF7 Klaviyo API] Headers: ' . print_r($args['headers'], true));
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('[CF7 Klaviyo API] WP Error: ' . $response->get_error_message());
            }
            return array(
                'success' => false,
                'error' => $response->get_error_message(),
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $decoded_body = json_decode($body, true);

        // Log de debug
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[CF7 Klaviyo API] Status Code: ' . $status_code);
            error_log('[CF7 Klaviyo API] Response Body: ' . $body);
        }

        if ($status_code >= 200 && $status_code < 300) {
            return array(
                'success' => true,
                'data' => $decoded_body,
                'status_code' => $status_code,
            );
        } else {
            // Mejorar el mensaje de error
            $error_message = $body;

            if (isset($decoded_body['errors']) && is_array($decoded_body['errors'])) {
                $error_message = $decoded_body['errors'];
            } elseif (isset($decoded_body['detail'])) {
                $error_message = $decoded_body['detail'];
            } elseif (isset($decoded_body['title'])) {
                $error_message = $decoded_body['title'];
            }

            return array(
                'success' => false,
                'error' => $error_message,
                'status_code' => $status_code,
                'raw_response' => $body,
            );
        }
    }

    /**
     * Test API connection
     */
    public function test_connection() {
        return $this->make_request('lists/');
    }

    /**
     * Get all lists from Klaviyo
     */
    public function get_lists() {
        $response = $this->make_request('lists/');

        if ($response['success']) {
            return array(
                'success' => true,
                'lists' => isset($response['data']['data']) ? $response['data']['data'] : array(),
            );
        }

        return $response;
    }

    /**
     * Subscribe a profile to a list
     */
    public function subscribe_to_list($list_id, $profile_data) {
        // Create or update profile first
        $profile_response = $this->create_or_update_profile($profile_data);

        if (!$profile_response['success']) {
            return $profile_response;
        }

        $profile_id = $profile_response['profile_id'];

        // Add profile to list
        $endpoint = 'list-relationships/';

        $data = array(
            'data' => array(
                array(
                    'type' => 'profile',
                    'id' => $profile_id,
                )
            )
        );

        // Subscribe to list using the relationships endpoint
        $subscribe_endpoint = "lists/{$list_id}/relationships/profiles/";

        return $this->make_request($subscribe_endpoint, 'POST', $data);
    }

    /**
     * Create or update a profile
     */
    public function create_or_update_profile($profile_data) {
        // Prepare profile data according to Klaviyo API v2024
        $data = array(
            'data' => array(
                'type' => 'profile',
                'attributes' => array()
            )
        );

        // Email is required
        if (!isset($profile_data['email']) || empty($profile_data['email'])) {
            return array(
                'success' => false,
                'error' => 'Email is required',
            );
        }

        $data['data']['attributes']['email'] = $profile_data['email'];

        // Add optional standard fields
        if (isset($profile_data['first_name'])) {
            $data['data']['attributes']['first_name'] = $profile_data['first_name'];
        }

        if (isset($profile_data['last_name'])) {
            $data['data']['attributes']['last_name'] = $profile_data['last_name'];
        }

        if (isset($profile_data['phone_number'])) {
            $data['data']['attributes']['phone_number'] = $profile_data['phone_number'];
        }

        // Add custom properties
        $custom_properties = array();
        $standard_fields = array('email', 'first_name', 'last_name', 'phone_number');

        foreach ($profile_data as $key => $value) {
            if (!in_array($key, $standard_fields) && !empty($value)) {
                $custom_properties[$key] = $value;
            }
        }

        if (!empty($custom_properties)) {
            $data['data']['attributes']['properties'] = $custom_properties;
        }

        $response = $this->make_request('profiles/', 'POST', $data);

        if ($response['success']) {
            return array(
                'success' => true,
                'profile_id' => $response['data']['data']['id'],
                'data' => $response['data'],
            );
        }

        return $response;
    }

    /**
     * Create a subscription in one call (profile + list)
     */
    public function create_subscription($list_id, $profile_data) {
        // Prepare the data for subscription creation
        $data = array(
            'data' => array(
                'type' => 'subscription',
                'attributes' => array(
                    'list_id' => $list_id,
                    'email' => $profile_data['email'],
                ),
                'relationships' => array(
                    'list' => array(
                        'data' => array(
                            'type' => 'list',
                            'id' => $list_id,
                        )
                    )
                )
            )
        );

        // Add profile properties
        $profile_properties = array();

        if (isset($profile_data['first_name'])) {
            $profile_properties['first_name'] = $profile_data['first_name'];
        }

        if (isset($profile_data['last_name'])) {
            $profile_properties['last_name'] = $profile_data['last_name'];
        }

        if (isset($profile_data['phone_number'])) {
            $profile_properties['phone_number'] = $profile_data['phone_number'];
        }

        // Add custom properties
        $standard_fields = array('email', 'first_name', 'last_name', 'phone_number');
        foreach ($profile_data as $key => $value) {
            if (!in_array($key, $standard_fields) && !empty($value)) {
                $profile_properties[$key] = $value;
            }
        }

        if (!empty($profile_properties)) {
            $data['data']['attributes']['profile'] = $profile_properties;
        }

        // Use the profile-subscription-bulk-create-jobs endpoint
        return $this->make_request('profile-subscription-bulk-create-jobs/', 'POST', $data);
    }
}
