<?php
/**
 * Form-specific Klaviyo settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF7_Klaviyo_Form_Settings {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_filter('wpcf7_editor_panels', array($this, 'add_klaviyo_panel'));
        add_action('wpcf7_save_contact_form', array($this, 'save_form_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_form_admin_assets'));

        // AJAX handlers
        add_action('wp_ajax_cf7_klaviyo_get_lists', array($this, 'ajax_get_lists'));
        add_action('wp_ajax_cf7_klaviyo_test_connection', array($this, 'ajax_test_connection'));
    }

    /**
     * Add Klaviyo panel to CF7 form editor
     */
    public function add_klaviyo_panel($panels) {
        $panels['klaviyo-panel'] = array(
            'title' => __('Klaviyo', 'cf7-klaviyo'),
            'callback' => array($this, 'render_klaviyo_panel'),
        );
        return $panels;
    }

    /**
     * Render the Klaviyo settings panel
     */
    public function render_klaviyo_panel($contact_form) {
        $form_id = $contact_form->id();
        $settings = $this->get_form_settings($form_id);

        // Get form fields
        $form_fields = $this->extract_form_fields($contact_form);

        ?>
        <h2><?php _e('Configuración de Klaviyo', 'cf7-klaviyo'); ?></h2>

        <fieldset>
            <legend>
                <label>
                    <input type="checkbox"
                           name="cf7_klaviyo[enabled]"
                           value="1"
                           <?php checked($settings['enabled'], true); ?>
                           id="cf7-klaviyo-enabled">
                    <?php _e('Activar integración con Klaviyo', 'cf7-klaviyo'); ?>
                </label>
            </legend>
        </fieldset>

        <div id="cf7-klaviyo-settings-container" style="<?php echo $settings['enabled'] ? '' : 'display:none;'; ?>">

            <hr>

            <fieldset>
                <legend><?php _e('Lista de Klaviyo', 'cf7-klaviyo'); ?></legend>

                <p class="description">
                    <?php _e('Selecciona la lista de Klaviyo donde se agregarán los contactos', 'cf7-klaviyo'); ?>
                </p>

                <p>
                    <button type="button" id="cf7-klaviyo-load-lists" class="button button-secondary">
                        <?php _e('Cargar Listas', 'cf7-klaviyo'); ?>
                    </button>
                    <span class="spinner" style="float: none;"></span>
                </p>

                <p id="cf7-klaviyo-list-select-container" style="<?php echo !empty($settings['list_id']) ? '' : 'display:none;'; ?>">
                    <select name="cf7_klaviyo[list_id]" id="cf7-klaviyo-list-id" class="regular-text">
                        <option value=""><?php _e('-- Selecciona una lista --', 'cf7-klaviyo'); ?></option>
                        <?php if (!empty($settings['list_id'])): ?>
                            <option value="<?php echo esc_attr($settings['list_id']); ?>" selected>
                                <?php echo esc_html($settings['list_name']); ?>
                            </option>
                        <?php endif; ?>
                    </select>
                </p>
            </fieldset>

            <hr>

            <fieldset>
                <legend><?php _e('Mapeo de Campos', 'cf7-klaviyo'); ?></legend>

                <p class="description">
                    <?php _e('Mapea los campos de tu formulario a las propiedades de Klaviyo. Los campos detectados automáticamente se basan en los tipos de campo de Contact Form 7.', 'cf7-klaviyo'); ?>
                </p>

                <table class="widefat cf7-klaviyo-field-mapping">
                    <thead>
                        <tr>
                            <th><?php _e('Campo del Formulario', 'cf7-klaviyo'); ?></th>
                            <th><?php _e('Tipo de Campo', 'cf7-klaviyo'); ?></th>
                            <th><?php _e('Propiedad Klaviyo', 'cf7-klaviyo'); ?></th>
                            <th><?php _e('Nombre Personalizado', 'cf7-klaviyo'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($form_fields)): ?>
                            <?php foreach ($form_fields as $field_name => $field_data): ?>
                                <?php
                                $saved_mapping = isset($settings['field_mapping'][$field_name]) ? $settings['field_mapping'][$field_name] : array();
                                $klaviyo_field = isset($saved_mapping['klaviyo_field']) ? $saved_mapping['klaviyo_field'] : $this->auto_detect_klaviyo_field($field_name);
                                $custom_name = isset($saved_mapping['custom_name']) ? $saved_mapping['custom_name'] : '';
                                ?>
                                <tr>
                                    <td>
                                        <strong><?php echo esc_html($field_name); ?></strong>
                                        <input type="hidden"
                                               name="cf7_klaviyo[field_mapping][<?php echo esc_attr($field_name); ?>][cf7_field]"
                                               value="<?php echo esc_attr($field_name); ?>">
                                        <input type="hidden"
                                               name="cf7_klaviyo[field_mapping][<?php echo esc_attr($field_name); ?>][field_type]"
                                               value="<?php echo esc_attr($field_data['type']); ?>">
                                    </td>
                                    <td>
                                        <code><?php echo esc_html($field_data['type']); ?></code>
                                    </td>
                                    <td>
                                        <select name="cf7_klaviyo[field_mapping][<?php echo esc_attr($field_name); ?>][klaviyo_field]"
                                                class="klaviyo-field-select">
                                            <option value=""><?php _e('-- No enviar --', 'cf7-klaviyo'); ?></option>
                                            <option value="email" <?php selected($klaviyo_field, 'email'); ?>>
                                                <?php _e('Email (requerido)', 'cf7-klaviyo'); ?>
                                            </option>
                                            <option value="first_name" <?php selected($klaviyo_field, 'first_name'); ?>>
                                                <?php _e('Nombre', 'cf7-klaviyo'); ?>
                                            </option>
                                            <option value="last_name" <?php selected($klaviyo_field, 'last_name'); ?>>
                                                <?php _e('Apellido', 'cf7-klaviyo'); ?>
                                            </option>
                                            <option value="phone_number" <?php selected($klaviyo_field, 'phone_number'); ?>>
                                                <?php _e('Teléfono', 'cf7-klaviyo'); ?>
                                            </option>
                                            <option value="custom" <?php selected($klaviyo_field, 'custom'); ?>>
                                                <?php _e('Propiedad Personalizada', 'cf7-klaviyo'); ?>
                                            </option>
                                        </select>
                                    </td>
                                    <td>
                                        <input type="text"
                                               name="cf7_klaviyo[field_mapping][<?php echo esc_attr($field_name); ?>][custom_name]"
                                               value="<?php echo esc_attr($custom_name); ?>"
                                               class="regular-text custom-field-name"
                                               placeholder="<?php _e('nombre_campo_personalizado', 'cf7-klaviyo'); ?>"
                                               <?php echo ($klaviyo_field === 'custom') ? '' : 'disabled'; ?>>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="4">
                                    <em><?php _e('No se detectaron campos en el formulario. Agrega campos en la pestaña "Formulario" y guarda antes de configurar Klaviyo.', 'cf7-klaviyo'); ?></em>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>

                <p class="description">
                    <strong><?php _e('Nota:', 'cf7-klaviyo'); ?></strong>
                    <?php _e('El campo Email es obligatorio para Klaviyo. Asegúrate de mapear al menos un campo como "Email".', 'cf7-klaviyo'); ?>
                </p>
            </fieldset>

            <hr>

            <fieldset>
                <legend><?php _e('Opciones Avanzadas', 'cf7-klaviyo'); ?></legend>

                <p>
                    <label>
                        <input type="checkbox"
                               name="cf7_klaviyo[double_opt_in]"
                               value="1"
                               <?php checked(isset($settings['double_opt_in']) && $settings['double_opt_in'], true); ?>>
                        <?php _e('Doble opt-in (enviar email de confirmación)', 'cf7-klaviyo'); ?>
                    </label>
                </p>

                <p>
                    <label>
                        <input type="checkbox"
                               name="cf7_klaviyo[log_errors]"
                               value="1"
                               <?php checked(isset($settings['log_errors']) ? $settings['log_errors'] : true, true); ?>>
                        <?php _e('Registrar errores en el log de WordPress', 'cf7-klaviyo'); ?>
                    </label>
                </p>
            </fieldset>

        </div>

        <?php wp_nonce_field('cf7_klaviyo_save_settings', 'cf7_klaviyo_nonce'); ?>

        <?php
    }

    /**
     * Extract form fields from Contact Form 7
     */
    private function extract_form_fields($contact_form) {
        $form_fields = array();
        $form_tags = $contact_form->scan_form_tags();

        foreach ($form_tags as $tag) {
            if (!empty($tag->name) && $tag->basetype !== 'submit') {
                $form_fields[$tag->name] = array(
                    'type' => $tag->basetype,
                    'label' => $tag->name,
                );
            }
        }

        return $form_fields;
    }

    /**
     * Auto-detect Klaviyo field based on CF7 field name
     */
    private function auto_detect_klaviyo_field($field_name) {
        $field_name_lower = strtolower($field_name);

        // Email detection
        if (strpos($field_name_lower, 'email') !== false || strpos($field_name_lower, 'correo') !== false) {
            return 'email';
        }

        // First name detection
        if (strpos($field_name_lower, 'first') !== false ||
            strpos($field_name_lower, 'nombre') !== false ||
            $field_name_lower === 'name') {
            return 'first_name';
        }

        // Last name detection
        if (strpos($field_name_lower, 'last') !== false ||
            strpos($field_name_lower, 'apellido') !== false ||
            strpos($field_name_lower, 'surname') !== false) {
            return 'last_name';
        }

        // Phone detection
        if (strpos($field_name_lower, 'phone') !== false ||
            strpos($field_name_lower, 'telefono') !== false ||
            strpos($field_name_lower, 'tel') !== false ||
            strpos($field_name_lower, 'movil') !== false) {
            return 'phone_number';
        }

        // Default to custom
        return 'custom';
    }

    /**
     * Save form settings
     */
    public function save_form_settings($contact_form) {
        if (!isset($_POST['cf7_klaviyo_nonce']) ||
            !wp_verify_nonce($_POST['cf7_klaviyo_nonce'], 'cf7_klaviyo_save_settings')) {
            return;
        }

        $form_id = $contact_form->id();
        $settings = isset($_POST['cf7_klaviyo']) ? $_POST['cf7_klaviyo'] : array();

        // Sanitize settings
        $sanitized_settings = array(
            'enabled' => isset($settings['enabled']) ? true : false,
            'list_id' => isset($settings['list_id']) ? sanitize_text_field($settings['list_id']) : '',
            'list_name' => isset($settings['list_name']) ? sanitize_text_field($settings['list_name']) : '',
            'field_mapping' => array(),
            'double_opt_in' => isset($settings['double_opt_in']) ? true : false,
            'log_errors' => isset($settings['log_errors']) ? true : false,
        );

        // Sanitize field mapping
        if (isset($settings['field_mapping']) && is_array($settings['field_mapping'])) {
            foreach ($settings['field_mapping'] as $field_name => $mapping) {
                $sanitized_settings['field_mapping'][$field_name] = array(
                    'cf7_field' => sanitize_text_field($mapping['cf7_field']),
                    'field_type' => sanitize_text_field($mapping['field_type']),
                    'klaviyo_field' => sanitize_text_field($mapping['klaviyo_field']),
                    'custom_name' => sanitize_text_field($mapping['custom_name']),
                );
            }
        }

        update_post_meta($form_id, '_cf7_klaviyo_settings', $sanitized_settings);
    }

    /**
     * Get form settings
     */
    public static function get_form_settings($form_id) {
        $default_settings = array(
            'enabled' => false,
            'list_id' => '',
            'list_name' => '',
            'field_mapping' => array(),
            'double_opt_in' => false,
            'log_errors' => true,
        );

        $saved_settings = get_post_meta($form_id, '_cf7_klaviyo_settings', true);

        if (!is_array($saved_settings)) {
            return $default_settings;
        }

        return array_merge($default_settings, $saved_settings);
    }

    /**
     * AJAX: Get Klaviyo lists
     */
    public function ajax_get_lists() {
        check_ajax_referer('cf7_klaviyo_admin', 'nonce');

        $api = new CF7_Klaviyo_API();
        $response = $api->get_lists();

        if ($response['success']) {
            $lists = array();
            foreach ($response['lists'] as $list) {
                $lists[] = array(
                    'id' => $list['id'],
                    'name' => $list['attributes']['name'],
                );
            }

            wp_send_json_success(array('lists' => $lists));
        } else {
            $error_message = $response['error'];

            // Si el error es un array (errores de Klaviyo API), formatearlo
            if (is_array($error_message)) {
                $formatted_errors = array();
                foreach ($error_message as $error) {
                    if (isset($error['detail'])) {
                        $formatted_errors[] = $error['detail'];
                    } elseif (isset($error['title'])) {
                        $formatted_errors[] = $error['title'];
                    } else {
                        $formatted_errors[] = print_r($error, true);
                    }
                }
                $error_message = implode('. ', $formatted_errors);
            }

            // Si sigue siendo un objeto/array, convertir a string
            if (!is_string($error_message)) {
                $error_message = print_r($error_message, true);
            }

            wp_send_json_error(array('message' => $error_message));
        }
    }

    /**
     * AJAX: Test connection
     */
    public function ajax_test_connection() {
        check_ajax_referer('cf7_klaviyo_test_connection', 'nonce');

        $api = new CF7_Klaviyo_API();
        $response = $api->test_connection();

        if ($response['success']) {
            wp_send_json_success(array('message' => __('Conexión exitosa con Klaviyo', 'cf7-klaviyo')));
        } else {
            $error_message = $response['error'];

            // Si el error es un array (errores de Klaviyo API), formatearlo
            if (is_array($error_message)) {
                $formatted_errors = array();
                foreach ($error_message as $error) {
                    if (isset($error['detail'])) {
                        $formatted_errors[] = $error['detail'];
                    } elseif (isset($error['title'])) {
                        $formatted_errors[] = $error['title'];
                    } else {
                        $formatted_errors[] = print_r($error, true);
                    }
                }
                $error_message = implode('. ', $formatted_errors);
            }

            // Si sigue siendo un objeto/array, convertir a string
            if (!is_string($error_message)) {
                $error_message = print_r($error_message, true);
            }

            // Agregar código de estado si existe
            if (isset($response['status_code'])) {
                $error_message = sprintf(
                    __('Error %d: %s', 'cf7-klaviyo'),
                    $response['status_code'],
                    $error_message
                );
            }

            wp_send_json_error(array('message' => $error_message));
        }
    }

    /**
     * Enqueue admin assets for form editor
     */
    public function enqueue_form_admin_assets($hook) {
        // Only load on CF7 form editor page
        if ($hook !== 'toplevel_page_wpcf7') {
            return;
        }

        wp_enqueue_script(
            'cf7-klaviyo-form-admin',
            CF7_KLAVIYO_PLUGIN_URL . 'assets/js/form-admin.js',
            array('jquery'),
            CF7_KLAVIYO_VERSION,
            true
        );

        wp_localize_script('cf7-klaviyo-form-admin', 'cf7KlaviyoForm', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cf7_klaviyo_admin'),
        ));

        wp_enqueue_style(
            'cf7-klaviyo-form-admin',
            CF7_KLAVIYO_PLUGIN_URL . 'assets/css/form-admin.css',
            array(),
            CF7_KLAVIYO_VERSION
        );
    }
}
