<?php
/**
 * Settings page for Klaviyo API configuration
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF7_Klaviyo_Settings {

    private static $instance = null;
    private $option_name = 'cf7_klaviyo_settings';

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    public function add_settings_page() {
        add_options_page(
            __('Configuración Klaviyo', 'cf7-klaviyo'),
            __('CF7 Klaviyo', 'cf7-klaviyo'),
            'manage_options',
            'cf7-klaviyo-settings',
            array($this, 'render_settings_page')
        );
    }

    public function register_settings() {
        register_setting(
            'cf7_klaviyo_settings_group',
            $this->option_name,
            array($this, 'sanitize_settings')
        );

        add_settings_section(
            'cf7_klaviyo_api_section',
            __('Configuración de API Klaviyo', 'cf7-klaviyo'),
            array($this, 'api_section_callback'),
            'cf7-klaviyo-settings'
        );

        add_settings_field(
            'api_key',
            __('API Key (Private Key)', 'cf7-klaviyo'),
            array($this, 'api_key_field_callback'),
            'cf7-klaviyo-settings',
            'cf7_klaviyo_api_section'
        );

        add_settings_field(
            'api_revision',
            __('API Revision', 'cf7-klaviyo'),
            array($this, 'api_revision_field_callback'),
            'cf7-klaviyo-settings',
            'cf7_klaviyo_api_section'
        );

        add_settings_field(
            'test_connection',
            __('Test de Conexión', 'cf7-klaviyo'),
            array($this, 'test_connection_field_callback'),
            'cf7-klaviyo-settings',
            'cf7_klaviyo_api_section'
        );
    }

    public function sanitize_settings($input) {
        $sanitized = array();

        if (isset($input['api_key'])) {
            $sanitized['api_key'] = sanitize_text_field($input['api_key']);
        }

        if (isset($input['api_revision'])) {
            $sanitized['api_revision'] = sanitize_text_field($input['api_revision']);
        } else {
            $sanitized['api_revision'] = '2024-10-15';
        }

        return $sanitized;
    }

    public function api_section_callback() {
        echo '<p>' . __('Configura tu API Key de Klaviyo para conectar Contact Form 7 con tus listas.', 'cf7-klaviyo') . '</p>';
        echo '<p>' . __('Puedes obtener tu API Key desde tu cuenta de Klaviyo en: Account > Settings > API Keys', 'cf7-klaviyo') . '</p>';
    }

    public function api_key_field_callback() {
        $settings = get_option($this->option_name);
        $api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
        ?>
        <input type="password"
               name="<?php echo $this->option_name; ?>[api_key]"
               value="<?php echo esc_attr($api_key); ?>"
               class="regular-text"
               placeholder="pk_xxxxxxxxxxxxxxxxxxxxx">
        <p class="description">
            <?php _e('Introduce tu Private API Key de Klaviyo', 'cf7-klaviyo'); ?>
        </p>
        <?php
    }

    public function api_revision_field_callback() {
        $settings = get_option($this->option_name);
        $api_revision = isset($settings['api_revision']) ? $settings['api_revision'] : '2024-10-15';
        ?>
        <input type="text"
               name="<?php echo $this->option_name; ?>[api_revision]"
               value="<?php echo esc_attr($api_revision); ?>"
               class="regular-text"
               placeholder="2024-10-15">
        <p class="description">
            <?php _e('Versión de la API de Klaviyo (por defecto: 2024-10-15)', 'cf7-klaviyo'); ?>
        </p>
        <?php
    }

    public function test_connection_field_callback() {
        ?>
        <button type="button" id="cf7-klaviyo-test-connection" class="button button-secondary">
            <?php _e('Probar Conexión', 'cf7-klaviyo'); ?>
        </button>
        <span id="cf7-klaviyo-test-result"></span>
        <?php
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_GET['settings-updated'])) {
            add_settings_error(
                'cf7_klaviyo_messages',
                'cf7_klaviyo_message',
                __('Configuración guardada correctamente', 'cf7-klaviyo'),
                'updated'
            );
        }

        settings_errors('cf7_klaviyo_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form action="options.php" method="post">
                <?php
                settings_fields('cf7_klaviyo_settings_group');
                do_settings_sections('cf7-klaviyo-settings');
                submit_button(__('Guardar Configuración', 'cf7-klaviyo'));
                ?>
            </form>

            <hr>

            <h2><?php _e('Guía de Uso', 'cf7-klaviyo'); ?></h2>
            <ol>
                <li><?php _e('Introduce tu API Key de Klaviyo arriba y guarda la configuración', 'cf7-klaviyo'); ?></li>
                <li><?php _e('Ve a Contact > Formularios de Contacto y edita el formulario que quieras integrar', 'cf7-klaviyo'); ?></li>
                <li><?php _e('En la pestaña "Klaviyo", activa la integración y selecciona la lista de destino', 'cf7-klaviyo'); ?></li>
                <li><?php _e('Mapea los campos del formulario a las propiedades de Klaviyo', 'cf7-klaviyo'); ?></li>
                <li><?php _e('¡Guarda y listo! Los envíos del formulario se sincronizarán con Klaviyo', 'cf7-klaviyo'); ?></li>
            </ol>
        </div>
        <?php
    }

    public function enqueue_admin_assets($hook) {
        if ('settings_page_cf7-klaviyo-settings' !== $hook) {
            return;
        }

        wp_enqueue_script(
            'cf7-klaviyo-admin',
            CF7_KLAVIYO_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            CF7_KLAVIYO_VERSION,
            true
        );

        wp_localize_script('cf7-klaviyo-admin', 'cf7KlaviyoAdmin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cf7_klaviyo_test_connection'),
        ));

        wp_enqueue_style(
            'cf7-klaviyo-admin',
            CF7_KLAVIYO_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            CF7_KLAVIYO_VERSION
        );
    }

    public static function get_api_key() {
        $settings = get_option('cf7_klaviyo_settings');
        return isset($settings['api_key']) ? $settings['api_key'] : '';
    }

    public static function get_api_revision() {
        $settings = get_option('cf7_klaviyo_settings');
        return isset($settings['api_revision']) ? $settings['api_revision'] : '2024-10-15';
    }
}
