<?php
/**
 * Handle Contact Form 7 submissions and send to Klaviyo
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF7_Klaviyo_Submission {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('wpcf7_before_send_mail', array($this, 'handle_submission'), 10, 1);
    }

    /**
     * Handle form submission
     */
    public function handle_submission($contact_form) {
        $form_id = $contact_form->id();
        $settings = CF7_Klaviyo_Form_Settings::get_form_settings($form_id);

        // Check if integration is enabled
        if (!$settings['enabled']) {
            return;
        }

        // Check if list is configured
        if (empty($settings['list_id'])) {
            $this->log_error($form_id, 'No list ID configured');
            return;
        }

        // Check if API key is configured
        $api_key = CF7_Klaviyo_Settings::get_api_key();
        if (empty($api_key)) {
            $this->log_error($form_id, 'No API key configured');
            return;
        }

        // Get submission data
        $submission = WPCF7_Submission::get_instance();
        if (!$submission) {
            $this->log_error($form_id, 'No submission instance');
            return;
        }

        $posted_data = $submission->get_posted_data();

        // Map fields to Klaviyo profile data
        $profile_data = $this->map_fields_to_klaviyo($posted_data, $settings['field_mapping']);

        // Validate email
        if (empty($profile_data['email'])) {
            $this->log_error($form_id, 'No email field mapped or email is empty');
            return;
        }

        // Send to Klaviyo
        $this->send_to_klaviyo($form_id, $settings['list_id'], $profile_data, $settings);
    }

    /**
     * Map CF7 fields to Klaviyo profile data
     */
    private function map_fields_to_klaviyo($posted_data, $field_mapping) {
        $profile_data = array();

        foreach ($field_mapping as $field_name => $mapping) {
            // Skip if field is not in posted data
            if (!isset($posted_data[$field_name])) {
                continue;
            }

            $value = $posted_data[$field_name];

            // Skip empty values
            if (empty($value) && $value !== '0') {
                continue;
            }

            // Handle arrays (checkboxes, multi-select)
            if (is_array($value)) {
                $value = implode(', ', $value);
            }

            // Clean value
            $value = $this->sanitize_field_value($value, $mapping['field_type']);

            // Skip if klaviyo_field is not set or empty
            if (empty($mapping['klaviyo_field'])) {
                continue;
            }

            // Map to standard or custom field
            if ($mapping['klaviyo_field'] === 'custom') {
                // Use custom name if provided
                if (!empty($mapping['custom_name'])) {
                    $profile_data[$mapping['custom_name']] = $value;
                }
            } else {
                // Standard field
                $profile_data[$mapping['klaviyo_field']] = $value;
            }
        }

        return $profile_data;
    }

    /**
     * Sanitize field value based on field type
     */
    private function sanitize_field_value($value, $field_type) {
        switch ($field_type) {
            case 'email':
                return sanitize_email($value);

            case 'url':
                return esc_url_raw($value);

            case 'number':
            case 'range':
                return floatval($value);

            case 'tel':
                // Keep only digits, plus, parentheses, and hyphens
                return preg_replace('/[^0-9+\-() ]/', '', $value);

            case 'textarea':
                return sanitize_textarea_field($value);

            default:
                return sanitize_text_field($value);
        }
    }

    /**
     * Send profile data to Klaviyo
     */
    private function send_to_klaviyo($form_id, $list_id, $profile_data, $settings) {
        $api = new CF7_Klaviyo_API();

        // Try to subscribe to list
        $response = $api->subscribe_to_list($list_id, $profile_data);

        if ($response['success']) {
            $this->log_success($form_id, $profile_data['email'], $list_id);

            // Store submission in WordPress for tracking
            $this->store_submission($form_id, $profile_data, $response);

        } else {
            $this->log_error(
                $form_id,
                'Failed to subscribe: ' . print_r($response['error'], true),
                $profile_data
            );
        }
    }

    /**
     * Store submission data
     */
    private function store_submission($form_id, $profile_data, $api_response) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'cf7_klaviyo_submissions';

        // Create table if it doesn't exist
        $this->maybe_create_table();

        $wpdb->insert(
            $table_name,
            array(
                'form_id' => $form_id,
                'email' => $profile_data['email'],
                'profile_data' => json_encode($profile_data),
                'klaviyo_response' => json_encode($api_response),
                'status' => 'success',
                'created_at' => current_time('mysql'),
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s')
        );
    }

    /**
     * Maybe create submissions table
     */
    private function maybe_create_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'cf7_klaviyo_submissions';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            form_id bigint(20) NOT NULL,
            email varchar(255) NOT NULL,
            profile_data text,
            klaviyo_response text,
            status varchar(50) NOT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY form_id (form_id),
            KEY email (email),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Log success
     */
    private function log_success($form_id, $email, $list_id) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf(
                '[CF7 Klaviyo] Successfully subscribed %s to list %s (Form ID: %d)',
                $email,
                $list_id,
                $form_id
            ));
        }
    }

    /**
     * Log error
     */
    private function log_error($form_id, $message, $data = null) {
        // Get form settings to check if logging is enabled
        $settings = CF7_Klaviyo_Form_Settings::get_form_settings($form_id);

        if (!$settings['log_errors']) {
            return;
        }

        $log_message = sprintf('[CF7 Klaviyo] Form ID %d - %s', $form_id, $message);

        if ($data) {
            $log_message .= ' | Data: ' . print_r($data, true);
        }

        error_log($log_message);

        // Store error in database
        global $wpdb;
        $table_name = $wpdb->prefix . 'cf7_klaviyo_submissions';

        $this->maybe_create_table();

        $wpdb->insert(
            $table_name,
            array(
                'form_id' => $form_id,
                'email' => isset($data['email']) ? $data['email'] : '',
                'profile_data' => $data ? json_encode($data) : '',
                'klaviyo_response' => $message,
                'status' => 'error',
                'created_at' => current_time('mysql'),
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s')
        );
    }
}
