<?php
/**
 * Panel de Pruebas de API Klaviyo
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF7_Klaviyo_Test_Panel {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('admin_menu', array($this, 'add_test_panel_page'), 99);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_test_assets'));

        // AJAX handlers
        add_action('wp_ajax_cf7_klaviyo_test_raw_request', array($this, 'ajax_test_raw_request'));
        add_action('wp_ajax_cf7_klaviyo_test_get_account', array($this, 'ajax_test_get_account'));
        add_action('wp_ajax_cf7_klaviyo_test_create_profile', array($this, 'ajax_test_create_profile'));
    }

    public function add_test_panel_page() {
        add_submenu_page(
            'options-general.php',
            __('Klaviyo Test Panel', 'cf7-klaviyo'),
            __('Klaviyo Tests', 'cf7-klaviyo'),
            'manage_options',
            'cf7-klaviyo-test-panel',
            array($this, 'render_test_panel')
        );
    }

    public function render_test_panel() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $api_key = CF7_Klaviyo_Settings::get_api_key();
        $api_revision = CF7_Klaviyo_Settings::get_api_revision();

        ?>
        <div class="wrap cf7-klaviyo-test-panel">
            <h1><?php _e('Panel de Pruebas de Klaviyo', 'cf7-klaviyo'); ?></h1>

            <?php if (empty($api_key)): ?>
                <div class="notice notice-warning">
                    <p>
                        <strong><?php _e('Advertencia:', 'cf7-klaviyo'); ?></strong>
                        <?php _e('No has configurado tu API Key. Ve a', 'cf7-klaviyo'); ?>
                        <a href="<?php echo admin_url('options-general.php?page=cf7-klaviyo-settings'); ?>">
                            <?php _e('Configuración', 'cf7-klaviyo'); ?>
                        </a>
                        <?php _e('para configurarla primero.', 'cf7-klaviyo'); ?>
                    </p>
                </div>
            <?php endif; ?>

            <div class="cf7-klaviyo-test-info">
                <h2><?php _e('Información de Configuración', 'cf7-klaviyo'); ?></h2>
                <table class="widefat">
                    <tbody>
                        <tr>
                            <td><strong><?php _e('API Key:', 'cf7-klaviyo'); ?></strong></td>
                            <td>
                                <?php if ($api_key): ?>
                                    <code><?php echo substr($api_key, 0, 10) . '...' . substr($api_key, -5); ?></code>
                                    <span style="color: green;">✓ Configurada</span>
                                <?php else: ?>
                                    <span style="color: red;">✗ No configurada</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('API Revision:', 'cf7-klaviyo'); ?></strong></td>
                            <td><code><?php echo esc_html($api_revision); ?></code></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Base URL:', 'cf7-klaviyo'); ?></strong></td>
                            <td><code>https://a.klaviyo.com/api/</code></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('WordPress Version:', 'cf7-klaviyo'); ?></strong></td>
                            <td><code><?php echo get_bloginfo('version'); ?></code></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('PHP Version:', 'cf7-klaviyo'); ?></strong></td>
                            <td><code><?php echo PHP_VERSION; ?></code></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <hr>

            <!-- Test 1: Petición Básica GET /lists/ -->
            <div class="cf7-klaviyo-test-section">
                <h2>1. <?php _e('Test Básico: GET /lists/', 'cf7-klaviyo'); ?></h2>
                <p class="description">
                    <?php _e('Obtiene todas las listas de Klaviyo. Este es el test más básico para verificar que la API Key funciona.', 'cf7-klaviyo'); ?>
                </p>
                <button type="button" class="button button-primary cf7-test-btn" data-test="get_lists" <?php disabled(empty($api_key)); ?>>
                    <?php _e('Ejecutar Test', 'cf7-klaviyo'); ?>
                </button>
                <div class="cf7-test-result" id="test-get-lists"></div>
            </div>

            <hr>

            <!-- Test 2: Obtener Información de Cuenta -->
            <div class="cf7-klaviyo-test-section">
                <h2>2. <?php _e('Test: GET /accounts/', 'cf7-klaviyo'); ?></h2>
                <p class="description">
                    <?php _e('Obtiene información de tu cuenta de Klaviyo.', 'cf7-klaviyo'); ?>
                </p>
                <button type="button" class="button button-primary cf7-test-btn" data-test="get_account" <?php disabled(empty($api_key)); ?>>
                    <?php _e('Ejecutar Test', 'cf7-klaviyo'); ?>
                </button>
                <div class="cf7-test-result" id="test-get-account"></div>
            </div>

            <hr>

            <!-- Test 3: Crear Perfil de Prueba -->
            <div class="cf7-klaviyo-test-section">
                <h2>3. <?php _e('Test: POST /profiles/ (Crear Perfil)', 'cf7-klaviyo'); ?></h2>
                <p class="description">
                    <?php _e('Crea un perfil de prueba en Klaviyo.', 'cf7-klaviyo'); ?>
                </p>
                <table class="form-table">
                    <tr>
                        <th><label for="test-email"><?php _e('Email de prueba:', 'cf7-klaviyo'); ?></label></th>
                        <td>
                            <input type="email" id="test-email" class="regular-text"
                                   value="test@example.com" placeholder="test@example.com">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="test-first-name"><?php _e('Nombre:', 'cf7-klaviyo'); ?></label></th>
                        <td>
                            <input type="text" id="test-first-name" class="regular-text"
                                   value="Test" placeholder="John">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="test-last-name"><?php _e('Apellido:', 'cf7-klaviyo'); ?></label></th>
                        <td>
                            <input type="text" id="test-last-name" class="regular-text"
                                   value="User" placeholder="Doe">
                        </td>
                    </tr>
                </table>
                <button type="button" class="button button-primary cf7-test-btn" data-test="create_profile" <?php disabled(empty($api_key)); ?>>
                    <?php _e('Ejecutar Test', 'cf7-klaviyo'); ?>
                </button>
                <div class="cf7-test-result" id="test-create-profile"></div>
            </div>

            <hr>

            <!-- Test 4: Petición Raw Personalizada -->
            <div class="cf7-klaviyo-test-section">
                <h2>4. <?php _e('Test Personalizado: Petición Raw', 'cf7-klaviyo'); ?></h2>
                <p class="description">
                    <?php _e('Prueba cualquier endpoint de la API de Klaviyo.', 'cf7-klaviyo'); ?>
                </p>
                <table class="form-table">
                    <tr>
                        <th><label for="raw-method"><?php _e('Método:', 'cf7-klaviyo'); ?></label></th>
                        <td>
                            <select id="raw-method">
                                <option value="GET">GET</option>
                                <option value="POST">POST</option>
                                <option value="PATCH">PATCH</option>
                                <option value="DELETE">DELETE</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="raw-endpoint"><?php _e('Endpoint:', 'cf7-klaviyo'); ?></label></th>
                        <td>
                            <input type="text" id="raw-endpoint" class="regular-text"
                                   value="lists/" placeholder="lists/">
                            <p class="description"><?php _e('Ejemplo: lists/, profiles/, accounts/', 'cf7-klaviyo'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="raw-body"><?php _e('Body (JSON):', 'cf7-klaviyo'); ?></label></th>
                        <td>
                            <textarea id="raw-body" class="large-text code" rows="8" placeholder='{"data": {}}'></textarea>
                            <p class="description"><?php _e('Solo para POST, PATCH. Dejar vacío para GET.', 'cf7-klaviyo'); ?></p>
                        </td>
                    </tr>
                </table>
                <button type="button" class="button button-primary cf7-test-btn" data-test="raw_request" <?php disabled(empty($api_key)); ?>>
                    <?php _e('Ejecutar Petición', 'cf7-klaviyo'); ?>
                </button>
                <div class="cf7-test-result" id="test-raw-request"></div>
            </div>

            <hr>

            <!-- Información de ayuda -->
            <div class="cf7-klaviyo-test-help">
                <h2><?php _e('Documentación de la API', 'cf7-klaviyo'); ?></h2>
                <ul>
                    <li>
                        <a href="https://developers.klaviyo.com/en/reference/api_overview" target="_blank">
                            <?php _e('Klaviyo API Reference', 'cf7-klaviyo'); ?>
                        </a>
                    </li>
                    <li>
                        <a href="https://developers.klaviyo.com/en/reference/get_lists" target="_blank">
                            <?php _e('GET /lists/', 'cf7-klaviyo'); ?>
                        </a>
                    </li>
                    <li>
                        <a href="https://developers.klaviyo.com/en/reference/create_profile" target="_blank">
                            <?php _e('POST /profiles/', 'cf7-klaviyo'); ?>
                        </a>
                    </li>
                </ul>

                <h3><?php _e('Endpoints Comunes', 'cf7-klaviyo'); ?></h3>
                <ul>
                    <li><code>GET lists/</code> - Obtener todas las listas</li>
                    <li><code>GET accounts/</code> - Obtener información de la cuenta</li>
                    <li><code>GET profiles/</code> - Obtener perfiles</li>
                    <li><code>POST profiles/</code> - Crear un perfil</li>
                    <li><code>GET segments/</code> - Obtener segmentos</li>
                </ul>
            </div>
        </div>

        <style>
            .cf7-klaviyo-test-panel {
                max-width: 1200px;
            }
            .cf7-klaviyo-test-section {
                background: #fff;
                padding: 20px;
                margin: 20px 0;
                border: 1px solid #ccc;
                border-radius: 4px;
            }
            .cf7-test-result {
                margin-top: 15px;
                padding: 15px;
                background: #f5f5f5;
                border-left: 4px solid #ccc;
                display: none;
                max-height: 500px;
                overflow-y: auto;
            }
            .cf7-test-result.success {
                background: #d4edda;
                border-left-color: #28a745;
                display: block;
            }
            .cf7-test-result.error {
                background: #f8d7da;
                border-left-color: #dc3545;
                display: block;
            }
            .cf7-test-result.loading {
                background: #d1ecf1;
                border-left-color: #17a2b8;
                display: block;
            }
            .cf7-test-result pre {
                background: #fff;
                padding: 10px;
                border-radius: 3px;
                overflow-x: auto;
                font-size: 12px;
            }
            .cf7-klaviyo-test-info table {
                max-width: 800px;
            }
            .cf7-test-btn {
                margin-top: 10px;
            }
            .cf7-test-btn:disabled {
                opacity: 0.5;
                cursor: not-allowed;
            }
        </style>
        <?php
    }

    public function enqueue_test_assets($hook) {
        if ('settings_page_cf7-klaviyo-test-panel' !== $hook) {
            return;
        }

        wp_enqueue_script(
            'cf7-klaviyo-test-panel',
            CF7_KLAVIYO_PLUGIN_URL . 'assets/js/test-panel.js',
            array('jquery'),
            CF7_KLAVIYO_VERSION,
            true
        );

        wp_localize_script('cf7-klaviyo-test-panel', 'cf7KlaviyoTest', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cf7_klaviyo_test'),
        ));
    }

    /**
     * AJAX: Test raw request
     */
    public function ajax_test_raw_request() {
        check_ajax_referer('cf7_klaviyo_test', 'nonce');

        $method = isset($_POST['method']) ? sanitize_text_field($_POST['method']) : 'GET';
        $endpoint = isset($_POST['endpoint']) ? sanitize_text_field($_POST['endpoint']) : 'lists/';
        $body = isset($_POST['body']) ? $_POST['body'] : null;

        $data = null;
        if ($body && in_array($method, array('POST', 'PATCH', 'PUT'))) {
            $data = json_decode(stripslashes($body), true);
        }

        $api = new CF7_Klaviyo_API();

        // Usar reflexión para acceder al método privado
        $reflection = new ReflectionClass($api);
        $method_obj = $reflection->getMethod('make_request');
        $method_obj->setAccessible(true);

        $response = $method_obj->invoke($api, $endpoint, $method, $data);

        wp_send_json($response);
    }

    /**
     * AJAX: Test get account
     */
    public function ajax_test_get_account() {
        check_ajax_referer('cf7_klaviyo_test', 'nonce');

        $api = new CF7_Klaviyo_API();

        $reflection = new ReflectionClass($api);
        $method_obj = $reflection->getMethod('make_request');
        $method_obj->setAccessible(true);

        $response = $method_obj->invoke($api, 'accounts/', 'GET');

        wp_send_json($response);
    }

    /**
     * AJAX: Test create profile
     */
    public function ajax_test_create_profile() {
        check_ajax_referer('cf7_klaviyo_test', 'nonce');

        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : 'test@example.com';
        $first_name = isset($_POST['first_name']) ? sanitize_text_field($_POST['first_name']) : 'Test';
        $last_name = isset($_POST['last_name']) ? sanitize_text_field($_POST['last_name']) : 'User';

        $profile_data = array(
            'email' => $email,
            'first_name' => $first_name,
            'last_name' => $last_name,
        );

        $api = new CF7_Klaviyo_API();
        $response = $api->create_or_update_profile($profile_data);

        wp_send_json($response);
    }
}
